import { Request, Response } from 'express';
import { customerService } from './service';

function hasFeature(vendor: any, featureKey: string): boolean {
  return vendor?.features?.includes(featureKey);
}

export const customersController = {
  // GET /customers
  async list(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'customer_management')) {
      return res.status(403).json({ error: 'Customer management not available on your plan.' });
    }
    const { search, tag } = req.query;
    const { data, error } = await customerService.listCustomers(vendor.id, search, tag);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // POST /customers
  async addOrLink(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'customer_management')) {
      return res.status(403).json({ error: 'Customer management not available on your plan.' });
    }
    const { data, error } = await customerService.addOrLinkCustomer(vendor.id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.status(201).json({ success: true, data });
  },

  // GET /customers/:customerId
  async getProfile(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'customer_management')) {
      return res.status(403).json({ error: 'Customer management not available on your plan.' });
    }
    const { customerId } = req.params;
    const { data, error } = await customerService.getCustomerProfile(vendor.id, customerId);
    if (error) return res.status(404).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // PUT /customers/:customerId
  async update(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'customer_management')) {
      return res.status(403).json({ error: 'Customer management not available on your plan.' });
    }
    const { customerId } = req.params;
    const { data, error } = await customerService.updateVendorCustomer(vendor.id, customerId, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // POST /customers/:customerId/measurements
  async upsertMeasurements(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'customer_management')) {
      return res.status(403).json({ error: 'Customer management not available on your plan.' });
    }
    const { customerId } = req.params;
    const { data, error } = await customerService.upsertMeasurements(vendor.id, customerId, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // DELETE /customers/:customerId
  async unlink(req: any, res: Response) {
    const vendor = req.vendor;
    if (!vendor || !hasFeature(vendor, 'customer_management')) {
      return res.status(403).json({ error: 'Customer management not available on your plan.' });
    }
    const { customerId } = req.params;
    const { success, error } = await customerService.unlinkCustomer(vendor.id, customerId);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success });
  }
}; 